
/* Copyright (C) 2008 Monotype Imaging Inc. All rights reserved. */

/* Confidential Information of Monotype Imaging Inc. */

/* adf.c */

#include <math.h>
#include "fs_itype.h"

#ifdef FS_EDGE_TECH
#include "adf.h"
#include "adffixedmath.h"
#endif

#ifdef FS_EDGE_RENDER
#include "fs_graymap.h"
#include "fs_effects.h"

/****************************/
/* Public API functions     */
/****************************/

/**
* API function to set default CSM values:
* inside cutoff, outside cutoff, and gamma.
* The application of these values depends on the font
* and the CSM control flag.
*/
FS_LONG FS_set_csm(_DS_ FS_FIXED insideCutOff,
                    FS_FIXED outsideCutOff, FS_FIXED gamma)
{
    STATE.insideCutoff  = (ADF_F32)insideCutOff;
    STATE.outsideCutoff = (ADF_F32)outsideCutOff;
    STATE.gamma         = (ADF_F32)gamma;
    return STATE.error = SUCCESS;
}

FS_LONG FS_get_csm(_DS_ FS_USHORT index, FS_FIXED *insideCutOff,
                    FS_FIXED *outsideCutOff, FS_FIXED *gamma)
{
    TTF *ttf = (TTF *)(STATE.cur_lfnt->fnt);

    if ((STATE.flags & FLAGS_DEFAULT_CSM_ON) || (ttf == 0))
    {
        *insideCutOff  = STATE.insideCutoff;
        *outsideCutOff = STATE.outsideCutoff;
        *gamma         = STATE.gamma;
    }
    else
    {
        ADFRenderAttrs adfRenderAttrs;
        adfGetCSMValues(_PS_ ttf, index, &adfRenderAttrs);
        *insideCutOff = adfRenderAttrs.insideCutoff;
        *outsideCutOff = adfRenderAttrs.outsideCutoff;
        *gamma = adfRenderAttrs.gamma;
    }
    return STATE.error = SUCCESS;
}

FS_LONG FS_set_csm_adjustments(_DS_
                               FS_FIXED sharpnessOffset, FS_FIXED sharpnessSlope,
                               FS_FIXED thicknessOffset, FS_FIXED thicknessSlope)
{
    FS_FIXED S,T;

    STATE.sharpnessOffset  = sharpnessOffset;
    STATE.sharpnessSlope   = sharpnessSlope;
    STATE.thicknessOffset  = thicknessOffset;
    STATE.thicknessSlope   = thicknessSlope;

    S = (sharpnessOffset + STATE.lpm * sharpnessSlope);
    T = (thicknessOffset + STATE.lpm * thicknessSlope);

    STATE.insideCutoffAdj  = -(S + T);
    STATE.outsideCutoffAdj =  (S - T);

    return STATE.error = SUCCESS;
}

FS_LONG FS_get_csm_adjustments(_DS_
                               FS_FIXED *sharpnessOffset, FS_FIXED *sharpnessSlope,
                               FS_FIXED *thicknessOffset, FS_FIXED *thicknessSlope)
{
    *sharpnessOffset = STATE.sharpnessOffset;
    *sharpnessSlope  = STATE.sharpnessSlope;
    *thicknessOffset = STATE.thicknessOffset;
    *thicknessSlope  = STATE.thicknessSlope;
    return STATE.error = SUCCESS;
}

#ifdef FS_TRIM_SUBPIXEL
/****************************************************************/
/* remove leading  trailing blank rows for sub-pixel output */
static FS_GRAYMAP *trim_subpixel(_DS_ FS_GRAYMAP *gmap, FS_USHORT type)
{
    int fr,lr;
    int height,width,bpl;
    FS_BYTE *p, mingray=0;
    FS_SHORT bpp = gmap->bitsPerPixel;

    if (bpp != 16 && bpp != 32)
        return gmap;

    bpl = gmap->bpl;
    width = gmap->width;
    height = gmap->height;

    /* Check whether trimming is needed */
    /* first non-blank row */
    p = gmap->bits;
    for (fr=0; fr<height; fr++)
    {
        int c;
        for (c=0; c<bpl; c++)
        {
            if (*p++ > mingray)
                break;
        }
        if (c != bpl)
            break;
    }

    /* last non-blank row */
    p = gmap->bits + bpl * (height-1);
    for (lr = height-1; lr>=fr; lr--)
    {
        int c;
        for (c=0; c<bpl; c++)
        {
            if (p[c] > mingray)
                break;
        }
        if (c != bpl)
            break;
        p -= bpl;
    }

    /* recompute height */
    height = 1 + lr - fr;

    if (fr!=0 || height != gmap->height) /* trimmed */
    {
        /* allocate a new graymap */
        FS_GRAYMAP *new_map;
        FS_LONG size;
        FS_BYTE *new_p;
        FS_BYTE *gmap_p = gmap->bits + fr*bpl;

        size = offsetof(FS_GRAYMAP, bits);
        size += height * bpl;
#ifdef FS_MEM_DBG
        STATE.memdbgid = "FS_GRAYMAP";
#endif
        new_map = (FS_GRAYMAP *)FSS_calloc(_PS_ size); /* see ITP-1975 */

        if (new_map == 0)
        {
            /* not fatal */
            STATE.error = SUCCESS;
            return gmap;
        }

        /* copy settings */
        new_map->cache_ptr = 0;
        new_map->dx = gmap->dx;
        new_map->dy = gmap->dy;
        new_map->i_dx = gmap->i_dx;
        new_map->i_dy = gmap->i_dy;
        new_map->embedded = false;
        new_map->size = size;
        new_map->lo_x = gmap->lo_x;
        new_map->hi_y = gmap->hi_y - (FS_SHORT)fr;
        new_map->width = (FS_SHORT)width;
        new_map->height = (FS_SHORT)height;
        new_map->type = type;
        new_map->bitsPerPixel = bpp;
        new_map->bpl = (FS_SHORT)bpl;
        new_p = new_map->bits;
        SYS_MEMCPY(new_p, gmap_p, height*bpl); 
        FSS_free_char(_PS_ gmap);
        return new_map;
    }
    else
        return gmap;
}
#endif /* FS_TRIM_SUBPIXEL */

#endif /* FS_EDGE_RENDER */

/****************************/
/* Private ADF functions    */
/****************************/

/* Functions common to both FS_EDGE_HINTS and FS_EDGE_RENDER */
#ifdef FS_EDGE_TECH

/** Initialize the ADF subsystem */
FS_VOID startADF(_DS0_)
{
    STATE.libInst = ADFInitSystem(_PS0_);
    if (STATE.libInst == 0)
        return;

#ifdef FS_EDGE_RENDER
    /* Default CSM values; may be changed based on contents of the ADFH table */
    /* or by calling FS_set_csm                                               */
    FS_set_csm(_PS_ (FS_FIXED)  26214 /* 0.4*/,
                     (FS_FIXED)-36700 /*-0.56*/,
                     (FS_FIXED) 65536 /* 1.0*/ );
    FS_set_csm_adjustments(_PS_ 0,0,0,0);
#endif
}

/** Terminate the ADF subsystem */
FS_VOID stopADF(_DS0_)
{
    ADFTermSystem(STATE.libInst);
}

/**
* private functions to search the nocenter glyph list in the adfh table. 
*/
/****************************************************************/
/* find the value of FLOOR(log2(count))                         */
/* this value is used as an entry selector in binary search     */
static FS_USHORT entrySelector(FS_USHORT count)
{
    FS_USHORT es = 1;
    FS_ULONG one = 1;

    while(count >= (one<<es))
        es++;

    return es-1;
}

/* binary search the no center list                     */
/* the list is sorted in ascending order by glyph index */
static FS_SHORT searchNoCenterList(FS_BYTE *noCenterPtr, FS_USHORT numNoCenter, FS_ULONG id)
{
    if (numNoCenter < 32)
    {
        /* linear search */
        FS_USHORT i, noCenterIndex;
        register FS_USHORT *p = (FS_USHORT *)noCenterPtr;
        for (i=0; i<numNoCenter; i++)
        {
            noCenterIndex = GET_xWORD(p+i);
            if (id == noCenterIndex)
                return 1;
            if (id <  noCenterIndex)
                return 0;
        }
    }
    else
    {
        /* binary search */
        FS_USHORT sel, *phi;
        register int lo, hi, idx;
        register FS_USHORT *p = (FS_USHORT *)noCenterPtr;
        register FS_USHORT t;

        lo = 0;
        hi = numNoCenter-1;

        sel = entrySelector(numNoCenter);

        switch(sel)
        {
            case 16: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case 15: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case 14: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case 13: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case 12: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case 11: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case 10: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case  9: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case  8: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case  7: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case  6: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case  5: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case  4: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case  3: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;/* FALLTHROUGH */
            case  2: idx = (lo+hi)>>1; t = GET_xWORD(p+idx); if (t > id) hi = idx; else lo = idx;
                     break;
            default: 
                break;
        }

        phi = p + hi + 1;
        p += lo;
        t = GET_xWORD(p);
        if (t == id)
            return 1;

        /* now a small linear search to find glyph index if it exists */
        while(p < phi)
        {
            t = GET_xWORD(p);
            if (t == id)
                return 1;
            p++;
        }
    }
    return 0;
}

/**
* Set the ADF alignment zone processing type for a glyph from the 
* ADFH table in the font. 
*/
FS_VOID adfGetEdgeHintData(_DS_ TTF *ttf, FS_ULONG glyph_index, FS_USHORT *EdgeHintType,
                           FS_SHORT *noCenter, FS_SHORT *cvtstart,
                           FS_SHORT *numylines, FS_SHORT *isrighttoleft)
{
    FS_USHORT i,numCSMRanges, sizeOfCSMEntries, numAlzIxEntries;
    FS_USHORT startGlyphIndex, endGlyphIndex;
    FS_USHORT numNoCenter;
    FS_ULONG table_version;
    FS_BYTE *pAdfh = (FS_BYTE *)(ttf->adfh);
    FS_BYTE *temp;

    *EdgeHintType = ADF_GRID_FIT_NONE;
    *cvtstart = 0;
    *numylines = 0;
    *isrighttoleft = 0;
    *noCenter = 0;
    startGlyphIndex = 0xffff;

    if (!pAdfh)
    {
        return; /* not using ADF alignment zones */
    }

    table_version = GET_xLONG(pAdfh);

    /* use only supported table versions:                           */
    /* version 1.5 - MAZ for CJK, hinted glyphs for other languages */
    /* version 2.0 - MAZ for CJK, BAZ for other languages           */

    if (table_version == 0x15000)
    {
        FS_BYTE *noCenterPtr;
        *EdgeHintType = ADF_GRID_FIT_MAZ_PIXEL; /* default if glyph index not found */
        pAdfh += 4;
        numCSMRanges = GET_xWORD(pAdfh);
        pAdfh += 2;
        sizeOfCSMEntries = GET_xWORD(pAdfh);
        pAdfh += 2 + numCSMRanges * SIZEOFCSMRANGE + sizeOfCSMEntries;
        
        numNoCenter = GET_xWORD(pAdfh);

        noCenterPtr = pAdfh + 2;

        pAdfh += 2 + numNoCenter * 2;

        numAlzIxEntries = GET_xWORD(pAdfh);
        pAdfh += 2;

        /* search for glyph index within ALZIXRECORDs */
        for (i=0; i<numAlzIxEntries; i++)
        {
            temp = pAdfh + 2;
            endGlyphIndex = GET_xWORD(temp); /* ALZIXRECORD->endGlyphIndex */
            if (glyph_index > endGlyphIndex)
            {
                pAdfh += SIZEOFALZIXRECORD;
                continue;
            }
            /* found ALZIXRECORD for glyph_index */
            startGlyphIndex = GET_xWORD(pAdfh); /* ALZIXRECORD->startGlyphIndex */            
            break;
        }
        if ((startGlyphIndex > glyph_index))
        {
            *EdgeHintType = ADF_GRID_FIT_MAZ_PIXEL;
            *noCenter = searchNoCenterList(noCenterPtr, numNoCenter, glyph_index);
        }
        else
        {
            *EdgeHintType = ADF_GRID_FIT_NONE;
        }
        return;
    }
    else if (table_version == 0x20000)
    {
        FS_BYTE *noCenterPtr;
        FS_USHORT groupno = 0;
        FS_BYTE *groupPtr;
        FS_SHORT numGroups;
        *EdgeHintType = ADF_GRID_FIT_MAZ_PIXEL; /* default if glyph index not found */
        pAdfh += 4;
        numCSMRanges = GET_xWORD(pAdfh);
        pAdfh += 2;
        sizeOfCSMEntries = GET_xWORD(pAdfh);
        pAdfh += 2 + numCSMRanges * SIZEOFCSMRANGE + sizeOfCSMEntries;

        numNoCenter = GET_xWORD(pAdfh);

        noCenterPtr = pAdfh + 2;

        pAdfh += 2 + numNoCenter * 2;

        numAlzIxEntries = GET_xWORD(pAdfh);
        pAdfh += 2;

        groupPtr = pAdfh + numAlzIxEntries * SIZEOFALZIXRECORD;

        /* search for glyph index within ALZIXRECORDs */
        for (i=0; i<numAlzIxEntries; i++)
        {
            temp = pAdfh+2;
            endGlyphIndex = GET_xWORD(temp); /* ALZIXRECORD->endGlyphIndex */
            if (glyph_index > endGlyphIndex)
            {
                pAdfh += SIZEOFALZIXRECORD;
                continue;
            }

            /* found ALZIXRECORD for glyph_index */
            startGlyphIndex = GET_xWORD(pAdfh); /* ALZIXRECORD->startGlyphIndex */
            temp = pAdfh+4;
            groupno = GET_xWORD(temp); /* ALZIXRECORD->startValueIndex */
            break; 
        }
        if ((groupno == 0xFFFF) || (startGlyphIndex > glyph_index))
        {
            *EdgeHintType = ADF_GRID_FIT_MAZ_PIXEL;
            *noCenter = searchNoCenterList(noCenterPtr, numNoCenter, glyph_index);
            return;
        }
        else
        {
            *EdgeHintType = ADF_GRID_FIT_BAZ_PIXEL;

            numGroups = GET_xWORD(groupPtr);
            if (groupno > numGroups)
                groupno = 0;

            groupPtr += 2 + groupno * 8;
            *cvtstart = GET_xWORD(groupPtr);
            temp = groupPtr+2;
            *numylines = GET_xWORD(temp);
            temp = groupPtr+4;
            *isrighttoleft = GET_xWORD(temp);
            return;
        }
    }
    else
        STATE.error = ERR_TABLE_UNSUPPORTED;

    return;
}

#endif /* FS_EDGE_TECH */

/* Functions for FS_EDGE_RENDER only */
#ifdef FS_EDGE_RENDER

/**
* Set the default ADF rendering attributes, including
* default CSM values.
*
* Called from the API function FSS_set_font()
*
*/
FS_VOID adfSetDefaultRenderAttrs(ADFRenderAttrs  *adfRenderAttrs)
{
    /* set what we can of ADFRenderAttrs */
    adfRenderAttrs->penX = 0 /*0.0*/;
    adfRenderAttrs->penY = 0 /*0.0*/;
    adfRenderAttrs->pointSize = 0;
    adfRenderAttrs->dpi = 72;
    adfRenderAttrs->scaleX = 65536 /*1.0*/;
    adfRenderAttrs->scaleY = 65536 /*1.0*/;
    adfRenderAttrs->rotationPtX = 0;
    adfRenderAttrs->rotationPtY = 0;
    adfRenderAttrs->rotationAngle = 0;
    adfRenderAttrs->displayMode = ADF_REND_MODE_CRT;
    adfRenderAttrs->gridFitType = 0;
    adfRenderAttrs->outsideCutoff = 0;
    adfRenderAttrs->insideCutoff = 0;
    adfRenderAttrs->gamma = 1;
    adfRenderAttrs->useColorReduction = 0;
    adfRenderAttrs->colorReductionAmt = 0;
}

/**
* Set the CSM values in the ADF rendering attributes by extracting them
* from the ADFH table in the font. If CSM values are not found, use the
* default values in the FS_STATE.
* Note that CSM records are sorted by size and the size represents the lower
* limit of the size range.
*/
FS_VOID adfGetCSMValues(_DS_ TTF *ttf, FS_ULONG glyph_index, ADFRenderAttrs *adfRenderAttrs)
{
    /* use defaults in STATE unless found in ADFH table */
    adfRenderAttrs->insideCutoff = STATE.insideCutoff;
    adfRenderAttrs->outsideCutoff = STATE.outsideCutoff;
    adfRenderAttrs->gamma = STATE.gamma;

    /* look for values in the ADFH table */
    if (ttf->adfh)
    {
        FS_USHORT i,numRanges, numRecords;
        FS_ULONG table_version;
        FS_BYTE *pAdfh = (FS_BYTE *)(ttf->adfh);
        FS_BYTE *endRange;
        CSMRANGE range;
        CSMRECORD record;
        FS_USHORT offset;
        table_version = GET_xLONG(pAdfh);
        if (table_version == 0x15000 || table_version == 0x20000) /* only use supported table versions */
        {
            pAdfh += 4;
            numRanges = GET_xWORD(pAdfh);
            if (numRanges == 0)
                return; /* and use default values */

            pAdfh += 4; /* skip to start of CSM range array */
            endRange = pAdfh + numRanges * SIZEOFCSMRANGE;
            range.endGlyphIndex = GET_xWORD(pAdfh); /* pAdfh points to CSMRANGE */
            while((pAdfh < endRange) && (glyph_index > range.endGlyphIndex))
            {
                pAdfh += SIZEOFCSMRANGE; /* skip this CSM range */
                range.endGlyphIndex = GET_xWORD(pAdfh);
            }
            if (pAdfh == endRange)
                return; /* should not happen */
            pAdfh += 2; /* pAdfh points to offset */
            offset = GET_xWORD(pAdfh);

            /* point to CSMENTRY */
            pAdfh = (FS_BYTE *)(ttf->adfh) + SIZEOFADFHEADER + numRanges*SIZEOFCSMRANGE + offset;
            numRecords = GET_xWORD(pAdfh);
            if (numRecords == 0)
                return; /* and use default values */

            pAdfh += 2;
            for (i=0; i<numRecords; i++)
            {
                record.size = GET_xWORD(pAdfh); /* pAdfh points to CSMRECORD */
                if (record.size <= STATE.lpm)
                {
                    record.insideCutoff = GET_xLONG(pAdfh+2);
                    record.outsideCutoff = GET_xLONG(pAdfh+6);
                    record.gamma = GET_xLONG(pAdfh+10);
                    adfRenderAttrs->insideCutoff = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)record.insideCutoff);
                    adfRenderAttrs->outsideCutoff = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)record.outsideCutoff);
                    adfRenderAttrs->gamma = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)record.gamma);
                }
                else
                    break;

                pAdfh += SIZEOFCSMRECORD;
            }
        }
        else
            STATE.error = ERR_TABLE_UNSUPPORTED;
    }
    return;
}


/**
* Convert an iType scaling matrix into typographic parameters 
* and fill in scale-dependent ADFRenderAttrs as much as possible.
*
* Called from the API function FSS_set_scale()
*
*/
FS_VOID adfComputeTypographicScaling(_DS_ SFNT *sfnt, ADFRenderAttrs  *adfRenderAttrs)
{
    FS_FIXED xppm, yppm, tan_s;
    FS_LONG error;

    error = get_scale_inputs(sfnt->user_scale, &xppm, &yppm, &tan_s);
    if (error)
        STATE.error = error;

    /* set scale dependent parameters in ADFRenderAttrs */
    adfRenderAttrs->pointSize = (ADF_F32)yppm;
    adfRenderAttrs->rotationAngle = (ADF_F32)0; /* outline is already rotated */
}

/** copy ADF render attributes */
FS_VOID adfCopyRenderAttrs(ADFRenderAttrs  *srcRenderAttrs, ADFRenderAttrs  *dstRenderAttrs)
{
    SYS_MEMCPY(dstRenderAttrs, srcRenderAttrs,sizeof(ADFRenderAttrs));
}

/**----------------------------------------------------------------------------------
*   Make an ADFPath from an iType outline for given id.
*   If using this function to generate SAZ data (deprecated), the id MUST be a Unicode value.
*   Otherwise the id value is not used by ADF.
*-----------------------------------------------------------------------------------*/
ADFPath *make_ADF_path(_DS_ FS_OUTLINE *outl, ADF_U32 id, FS_GRAYMAP **g,
                        FS_FIXED *scaled_dx, FS_FIXED *scaled_dy)
{
    FS_BYTE *p;
    FS_LONG size, PenCmd_off;
    ADFPath* a = 0;

    *g = 0; /* no space graymap returned yet */

    *scaled_dx = outl->dx;
    *scaled_dy = outl->dy;

    /* space character */
    if (outl->num == 0)
    {
        FS_GRAYMAP *gmap;

        size = sizeof(FS_GRAYMAP) - 1;
#ifdef FS_MEM_DBG
        STATE.memdbgid = "FS_GRAYMAP";
#endif
        gmap = (FS_GRAYMAP *) FSS_calloc(_PS_ size);
        if (gmap == 0)
            return 0;
        gmap->size = size;
        gmap->dx = *scaled_dx;
        gmap->dy = *scaled_dy;
        /* Set integer escapements */
        if (gmap->dx == 0 || gmap->dy == 0)
        {
            /* rotation of N*90 degrees, integer escapements are useful */
            gmap->i_dx = FS_ROUND(gmap->dx); /* rounding here */
            gmap->i_dy = FS_ROUND(gmap->dy); /* rounding here */
        }
        else /* other rotations, integer escapements are not useful */
            gmap->i_dx = gmap->i_dy = 0;
        gmap->embedded = false;        /* this graymap was outline-generated */
        *g = gmap;
#ifdef FS_STATS
        made_gmap++;
#endif
        return 0;  /* no ADF path made */
    }

    /* compute size of ADFpath structure and all data */
    size = sizeof(ADFPath);
    FS_ALIGN(size);
    PenCmd_off = size;
    size += outl->num * sizeof(ADFPenCmd);
    if (!outl->num)
        size = size;

    /* allocate an ADFpath and data and fill it in */
#ifdef FS_MEM_DBG
    STATE.memdbgid = "ADFPath+data";
#endif
    p = (FS_BYTE *)FSS_calloc(_PS_ size);
    if (p)
    {
        ADFPenCmd* pc;
        FS_FIXED*  x;
        FS_FIXED*  y;
        FS_SHORT num;
        FS_BYTE* type;
        FS_FIXED  prev_x=0;
        FS_FIXED  prev_y=0;
        FS_BYTE prev_type=0;
        FS_SHORT not_first;
        ADFPenCmd* prev_pc;
        FS_FIXED sw_half = (FIXEDTODOT6((FS_FIXED)(STATE.stroke_pct * STATE.lpm))/2) << 10;
        FS_BOOLEAN dot_to_line = 0;
#ifdef FS_STIK
        int is_stik;
#endif

        a = (ADFPath*) p;
        a->penCmds = (ADFPenCmd*)(p + PenCmd_off);

        /* em box size */
        a->fontUnitsPerEM = ((ADF_F32)STATE.lpm)<<16;  

        /* character code */
        a->charCode = id;

        /*lint -e571  Warning 571: Warning -- Suspicious cast */
        /* number of contours */
        a->numContours = (ADF_U32)outl->nc;

        /* number of pen commands */
        a->numPenCmds = (ADF_U32)outl->num;

        /*lint +e571  Warning 571: Warning -- Suspicious cast */
#ifdef FS_STIK
        /* ? is this a stik character */
        if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) && !(((FS_BYTE *)(outl->type))[0] & OUTLINE_CHAR))
        {
            a->pathType = ADF_UNIFORM_STROKE_PATH;
            a->pathWidth = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)STATE.stroke_pct);
            a->pathWidth *= STATE.lpm;
            /* use an integer stroke width if raster special effects apply */
            if (STATE.flags & (FLAGS_EMBOSSED | FLAGS_ENGRAVED | FLAGS_OUTLINED | FLAGS_OUTLINED_2PIXEL | \
                               FLAGS_OUTLINED_UNFILLED))
            {
                FS_FIXED fsw = MAX(FIXED_ONE,a->pathWidth);
                a->pathWidth = (fsw + FIXED_ONEHALF) & 0xFFFF0000;
            }
        }
        else
#endif /* FS_STIK */
        {
            a->pathType = ADF_OUTLINE_PATH;
            a->pathWidth = 0;
        }

        /* Convert type and outline data */
        pc = a->penCmds;

        num  = outl->num;
        type = (FS_BYTE *)( outl->type);
        x    = (FS_FIXED *)(outl->x);
        y    = (FS_FIXED *)(outl->y);

        /* Bounding box */
        a->glyphMinX = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*x);
        a->glyphMinY = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*y);
        a->glyphMaxX = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*x);
        a->glyphMaxY = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*y);

#ifdef FS_STIK
        is_stik = (a->pathType == ADF_UNIFORM_STROKE_PATH);
        if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) && is_stik)
        {
#ifdef FS_PSEUDO_BOLD
            if (STATE.bold_pct || STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct)
            {
                if ( (STATE.flags & (FLAGS_EMBOSSED | FLAGS_ENGRAVED | FLAGS_OUTLINED_1PIXEL | 
                                    FLAGS_OUTLINED_2PIXEL | FLAGS_OUTLINED_UNFILLED)) ||
                    ( !(STATE.flags & FLAGS_ADD_WEIGHT) &&
                    ( !( STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) || STATE.lpm > 26 ) ) 
                )
                    dot_to_line = 0;
            }
            else
#endif /* FS_PSEUDO_BOLD */
            {
                if ( !( STATE.cur_lfnt->fontflags & FONTFLAG_EDGE) &&
                     !( STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON))
                    dot_to_line = 1;
            }
        }
#endif /* FS_STIK */

        not_first = 0;
        prev_pc = pc;

        while (num--)
        {
            /* Convert one drawing element */
            pc->opCode = 3 & (*type);  /* iType and ADF use the same numerical values */
                                         /* iType sometimes uses the higher type bits   */

            if (pc->opCode == ADF_PEN_CURVTO_CMD)
            {
                pc->cx = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*x); prev_x = *x; x++;
                pc->cy = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*y); prev_y = *y; y++;
            }
            if (pc->opCode == ADF_PEN_CUBETO_CMD)
            {
                pc->cx = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*x); x++;
                pc->cy = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*y); y++;
                pc->cx2 = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*x); prev_x = *x; x++;
                pc->cy2 = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*y); prev_y = *y; y++;
            }

            pc->x = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*x);
            pc->y = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)*y);
            if (not_first)
            {
                if ((prev_x == *x) && (prev_y == *y))
                {
                    if (dot_to_line)
                    {
                        if ((prev_type == 0) && ((*type) == 1))
                        {
                            pc->x = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)(*x));
                            pc->y = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)(*y+sw_half));
                            prev_pc->x = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)(*x));
                            prev_pc->y = (ADF_F32)I1616_TO_FLOAT((ADF_I1616)(*y-sw_half));
                        }
                    }
                }
            }

            not_first = 1;
            prev_x = *x;
            x++;
            prev_y = *y;
            y++;
            prev_type = *type;
            type++;

            if (pc->x < a->glyphMinX)
                a->glyphMinX = pc->x;
            if (pc->x > a->glyphMaxX)
                a->glyphMaxX = pc->x;
            if (pc->y < a->glyphMinY)
                a->glyphMinY = pc->y;
            if (pc->y > a->glyphMaxY)
                a->glyphMaxY = pc->y;

            prev_pc = pc;
            pc +=1;
        }
    }
    return a;
}

/**----------------------------------------------------------------------------------
*    Make an ADFGlyph from an ADFPath. Returns an opaque pointer to the ADFGlyph.
*-----------------------------------------------------------------------------------*/
static FS_VOID* make_ADF_char(_DS_ ADFPath* adfPath)
{
    void* c;

    if (!adfPath)
        return 0;

    /* make an adf glyph from the path  */
    /* returns opaque pointer to an ADF */
    c = ADFGenerateADF(STATE.libInst, adfPath);
    FSS_free(_PS_ adfPath);
    return c;
}

/**----------------------------------------------------------------------------------
*    Setup ADF rendering and render an ADF density image from an ADFGlyph.
*-----------------------------------------------------------------------------------*/
static ADFImage* make_ADF_density_image(_DS_ ADFGlyph* c,
                                        ADFRenderImageAttrs *adfRenderImageAttrs,
                                        ADFTypesetAttrs *adfTypesetAttrs)
{
    ADFImage* di;
    ADF_U16 type;
    ADFRenderGlyphData adfRenderGlyphData;
    ADFRenderAttrs adfRenderAttrs;

    if (!c)
        return 0;

    adfCopyRenderAttrs(&STATE.cur_typeset.tfntarray[STATE.cur_font].adfRenderAttrs, 
                       &adfRenderAttrs);

    if (STATE.flags & FLAGS_DEFAULT_CSM_ON)
    {    /* override font values in STATE copy */
        adfRenderAttrs.insideCutoff = STATE.insideCutoff + STATE.insideCutoffAdj;
        adfRenderAttrs.outsideCutoff = STATE.outsideCutoff + STATE.outsideCutoffAdj;
        adfRenderAttrs.gamma = STATE.gamma;
    }

    if (adfRenderAttrs.insideCutoff < adfRenderAttrs.outsideCutoff)
    {   /* clamp cutoffs to prevent inverted cutoffs */
        if (adfRenderAttrs.outsideCutoff < 0)
        {   /* thick binary output */
            adfRenderAttrs.insideCutoff = adfRenderAttrs.outsideCutoff;
        }
        else
        {   /* regular binary output */
            adfRenderAttrs.insideCutoff = 0;
            adfRenderAttrs.outsideCutoff = 0;
        }
    }

    adfRenderAttrs.displayMode = adfRenderImageAttrs->displayMode;

    /* Use adfRenderAttrs to fill in the other 3 structures */
    ADFRenderSetup(c,
                   &adfRenderAttrs,
                   &adfRenderGlyphData,
                   adfRenderImageAttrs,
                   adfTypesetAttrs);

    if (adfRenderAttrs.displayMode == ADF_REND_MODE_CRT)
        type = ADF_IMAGE_TYPE_GRAY;
    else
        type = ADF_IMAGE_TYPE_RGBA;

    /* create an empty density image structure */
    di = ADFCreateImage(STATE.libInst, type,
                        adfRenderImageAttrs->imageW,
                        adfRenderImageAttrs->imageH, 0);

    /* fill in the density image structure */
    if (di)
        ADFRenderGlyph(STATE.libInst, c, &adfRenderGlyphData, di);

    ADFDestroyADF(STATE.libInst, c);
    if (STATE.error)
    {
        ADFDestroyImage(STATE.libInst, di);
        di = NULL;
    }

    return di;
}

/**----------------------------------------------------------------------------------
*    Convert a density image into an iType 8-bit graymap or subpixel graymap.
*-----------------------------------------------------------------------------------*/
static FS_GRAYMAP* convert_to_graymap(_DS_ ADFImage* di, FS_FIXED scaled_dx, FS_FIXED scaled_dy,
                                      ADFRenderImageAttrs *adfRenderImageAttrs,
                                      ADFTypesetAttrs *adfTypesetAttrs, FS_USHORT type)
{
    FS_LONG size, imageSize;
    FS_SHORT bytesPerPixel;
    FS_GRAYMAP* g;
    int bplshift=0;
    int bplceil=0;
    FS_SHORT bpp=4;

    if (!di)
        return 0;
    if (adfRenderImageAttrs->displayMode == ADF_REND_MODE_CRT)
        bytesPerPixel = 1;
    else
        bytesPerPixel = 4;

    if (type & FS_MAP_EDGE_GRAYMAP4)
    {
        bplshift = 1;
        bplceil = 1;
        bpp = 4;
        type = FS_MAP_EDGE_GRAYMAP4;
    }
    else if (type & FS_MAP_EDGE_GRAYMAP8)
    {
        bplshift = 0;
        bplceil = 0;
        bpp = 8;
        type = FS_MAP_EDGE_GRAYMAP8;
    }
    else if (type & FS_MAP_EDGE_GRAYMAP2)
    {
        bplshift = 2;
        bplceil = 3;
        bpp = 2;
        type = FS_MAP_EDGE_GRAYMAP2;
    }
    else if (type & FS_MAP_EDGE_RGBv4)
    {
        bplshift = 1;
        bplceil = 1;
        bpp = 4;
        type = FS_MAP_EDGE_RGBv4;
    }
    else if (type & FS_MAP_EDGE_RGBv8)
    {
        bplshift = 0;
        bplceil = 0;
        bpp = 8;
        type = FS_MAP_EDGE_RGBv8;
    }
    else if (type & FS_MAP_EDGE_RGBh4)
    {
        bplshift = 1;
        bplceil = 1;
        bpp = 4;
        type = FS_MAP_EDGE_RGBh4;
    }
    else if (type & FS_MAP_EDGE_RGBh8)
    {
        bplshift = 0;
        bplceil = 0;
        bpp = 8;
        type = FS_MAP_EDGE_RGBh8;
    }

    size = sizeof(FS_GRAYMAP);
    imageSize = bytesPerPixel*di->w * di->h;
    size += imageSize - 1;   /* because 1 byte is allocated with the structure */

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_GRAYMAP";
#endif
    g = (FS_GRAYMAP*)FSS_calloc(_PS_ size);
    if (g)
    {
        FS_LONG  row, w, h;

        g->size = size;
        g->embedded = 0;
        g->width = di->w;  /* application must know how many bytes per pixel    */
                           /* instead of iType's 2 pixels per byte              */
        g->height = di->h;
        g->bpl = (bytesPerPixel*di->w + (ADF_U16)bplceil)>>bplshift;
        g->bitsPerPixel = bytesPerPixel * bpp;
        g->type = type;

        g->lo_x = (FS_SHORT)adfRenderImageAttrs->imageOriginX;
        g->hi_y = (FS_SHORT)adfRenderImageAttrs->imageOriginY + di->h;
        if (STATE.adfGridFitType == ADF_GRID_FIT_NONE)
        {
            g->dx = scaled_dx;
            g->dy = scaled_dy;
        }
        else
        {
            if (scaled_dx && scaled_dy)
            {
                /* only adjust rotated glyphs */
                g->dx = FixMul(scaled_dx,adfTypesetAttrs->adjFUToPixelScaleX) +
                        (FS_FIXED)(adfTypesetAttrs->gridAlgnAdjX);
                g->dy = FixMul(scaled_dy,adfTypesetAttrs->adjFUToPixelScaleY) +
                        (FS_FIXED)(adfTypesetAttrs->gridAlgnAdjY);
            }
            else
            {
                g->dx = scaled_dx;
                g->dy = scaled_dy;
            }
        }

        /* Set integer escapements */
        if (ABS(g->dx) <= 1 || ABS(g->dy)<=1)
        {
            /* rotation of N*90 degrees, integer escapements are useful */
            g->i_dx = FS_ROUND(g->dx); /* rounding here */
            g->i_dy = FS_ROUND(g->dy); /* rounding here */
        }
        else /* other rotations, integer escapements are not useful */
            g->i_dx = g->i_dy = 0;

        /* ADF images and iType graymaps are upside down from */
        /* each other. Flip ADF image into iType graymap      */
        w = bytesPerPixel*di->w;
        h = di->h;
        if (bytesPerPixel == 1) /* regular graymap */
        {
            if (g->bitsPerPixel == 4)
            {
                ADF_U8*  src;
                FS_BYTE* dst;
                FS_ULONG hi,lo;
                int bpl;
                FS_LONG m;

                m = w & 1;
                bpl = g->bpl - m;
                src = di->base + (h-1)*w;
                dst = g->bits;
                for (row = 0; row < h; row++)
                {
                    FS_LONG col;
                    col = bpl;
                    while (col--)
                    {
                        hi = (*src++) & 0xf0;
                        lo = (*src++) >> 4;
                        *dst++ = (FS_BYTE)(hi | lo);
                    }

                    if (m)
                    {
                        hi = (FS_BYTE)(*src++) & 0xf0;
                        *dst++ = (FS_BYTE)(hi);
                    }
                    src -= 2*w; /* adjust by 2 rows */
                }
            }
            else if (g->bitsPerPixel == 8 )
            {
                ADF_U8*  src;
                FS_BYTE* dst;
                dst = &g->bits[0];
                src = di->base + (h - 1) * w;
                for (row = 0; row < h; row++)
                {
                    SYS_MEMCPY(dst, src, w);
                    src -= w;
                    dst += w;
                }
            }
            else if (g->bitsPerPixel == 2)
            {
                ADF_U8*  src;
                FS_BYTE* dst;
                FS_BYTE a,b,c,d,z;
                int bpl;
                FS_LONG m;

                m = w & 3;
                if (m == 0)
                    bpl = g->bpl;
                else
                    bpl = g->bpl - 1;

                src = di->base + (h-1)*w;
                dst = g->bits;

                for (row = 0; row < h; row++)
                {
                    FS_LONG col;
                    for (col = 0; col<bpl; col++)
                    {
                        a = (FS_BYTE)((*src++) >> 6);
                        b = (FS_BYTE)((*src++) >> 6);
                        c = (FS_BYTE)((*src++) >> 6);
                        d = (FS_BYTE)((*src++) >> 6);

                        z = (a<<6) | (b<<4) | (c<<2) | d;
                        *dst++ = z;
                    }

                    switch (m)
                    {
                    case 0: break;
                    case 1:
                        {
                             a = (FS_BYTE)((*src++) >> 6);
                             z = (a<<6); 
                             *dst++ = z;
                             break;
                        }
                    case 2:
                        {
                            a = (FS_BYTE)((*src++) >> 6);
                            b = (FS_BYTE)((*src++) >> 6);
                            z = (a<<6) | (b<<4); 
                            *dst++ = z;
                            break;
                        }
                    case 3:
                        {
                            a = (FS_BYTE)((*src++) >> 6);
                            b = (FS_BYTE)((*src++) >> 6);
                            c = (FS_BYTE)((*src++) >> 6);
                            z = (a<<6) | (b<<4)| (c<<2);
                            *dst++ = z;
                            break;
                        }
                    default: break;
                    }
                    src -= 2*w; /* adjust by 2 rows */
                }
            }
        }
        else /* subpixel graymap */
        {
            if (g->bitsPerPixel == 16)
            {
                ADF_U8*  src;
                FS_BYTE* dst;
                FS_ULONG hi,lo;

                src = di->base + (h-1)*w;
                dst = g->bits;
                for (row = 0; row < h; row++)
                {
                    FS_LONG col;
                    col = g->bpl;
                    while (col)
                    {
                        hi = (*src++) & 0xf0;
                        lo = (*src++) >> 4;
                        *dst++ = (FS_BYTE)(hi | lo);
                        hi = (*src++) & 0xf0;
                        lo = (*src++) >> 4;
                        *dst++ = (FS_BYTE)(hi | lo);
                        col -= 2;
                    }

                    src -= 2*w; /* adjust by 2 rows */
                }
            }
            else if (g->bitsPerPixel == 32 )
            {
                ADF_U8*  src;
                FS_BYTE* dst;
                dst = &g->bits[0];
                src = di->base + (h - 1) * w;
                for (row = 0; row < h; row++)
                {
                    SYS_MEMCPY(dst, src, w);
                    src -= w;
                    dst += w;
                }
            }
            else if (g->bitsPerPixel == 8)
            {
                ADF_U8*  src;
                FS_BYTE* dst;
                FS_BYTE a,b,c,d,z;

                src = di->base + (h-1)*w;
                dst = g->bits;
                for (row = 0; row < h; row++)
                {
                    FS_LONG col;
                    col = g->bpl;
                    while(col--)
                    {
                        a = (FS_BYTE)((*src++) >> 6);
                        b = (FS_BYTE)((*src++) >> 6);
                        c = (FS_BYTE)((*src++) >> 6);
                        d = (FS_BYTE)((*src++) >> 6);

                        z = (a<<6) | (b<<4) | (c<<2) | d;
                        *dst++ = z;
                    }
                    src -= 2*w; /* adjust by 2 rows */
                }
            }
        }
    } /* if (g) */

    ADFDestroyImage(STATE.libInst, di);
    return g;
}


/**---------------------------------------------------------------------------
*    Convert a density image into an iType 8-bit graymap containing 
*    distance field. 
*-----------------------------------------------------------------------------*/
static FS_GRAYMAP* convert_to_distance_field(_DS_ ADFImage* di,
                                      FS_FIXED scaled_dx, FS_FIXED scaled_dy,
                                      ADFRenderImageAttrs *adfRenderImageAttrs,
                                      ADFTypesetAttrs *adfTypesetAttrs)
{
    FS_LONG size, imageSize;
    FS_GRAYMAP* g;

    size = sizeof(FS_GRAYMAP);
    imageSize = di->w * di->h;
    size += imageSize - 1;   /* because 1 byte is allocated with the structure */

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_GRAYMAP";
#endif
    g = (FS_GRAYMAP*)FSS_calloc(_PS_ size);
    if (g)
    {
        FS_LONG  row, w, h;

        g->size = size;
        g->embedded = 0;
        g->width = di->w;  /* application must know how many bytes per pixel    */
                           /* instead of iType's 2 pixels per byte              */
        g->height = di->h;
        g->bpl = di->w;
        g->bitsPerPixel = 8;
        g->type = FS_MAP_GRAYMAP8;

        g->lo_x = (FS_SHORT)adfRenderImageAttrs->imageOriginX;
        g->hi_y = (FS_SHORT)adfRenderImageAttrs->imageOriginY + di->h;
        if (STATE.adfGridFitType == ADF_GRID_FIT_NONE)
        {
            g->dx = scaled_dx;
            g->dy = scaled_dy;
        }
        else
        {
            if (scaled_dx && scaled_dy)
            {
                /* only adjust rotated glyphs */
                g->dx = FixMul(scaled_dx,adfTypesetAttrs->adjFUToPixelScaleX) +
                        (FS_FIXED)(adfTypesetAttrs->gridAlgnAdjX);
                g->dy = FixMul(scaled_dy,adfTypesetAttrs->adjFUToPixelScaleY) +
                        (FS_FIXED)(adfTypesetAttrs->gridAlgnAdjY);
            }
            else
            {
                g->dx = scaled_dx;
                g->dy = scaled_dy;
            }
        }

        /* Set integer escapements */
        if (ABS(g->dx) <= 1 || ABS(g->dy)<=1)
        {
            /* rotation of N*90 degrees, integer escapements are useful */
            g->i_dx = FS_ROUND(g->dx); /* rounding here */
            g->i_dy = FS_ROUND(g->dy); /* rounding here */
        }
        else /* other rotations, integer escaepments are not useful */
            g->i_dx = g->i_dy = 0;

        /* ADF images and iType graymaps are upside down from */
        /* each other. Flip ADF image into iType graymap      */
        w = di->w;
        h = di->h;
        {
            ADF_U8*  src;
            FS_BYTE* dst;
            dst = &g->bits[0];
            src = di->base + (h - 1) * w;
            for (row = 0; row < h; row++)
            {
                SYS_MEMCPY(dst, src, w);
                src -= w;
                dst += w;
            }
        }
    }

    ADFDestroyImage(STATE.libInst, di);
    return g;
}

/**
* Private function to make a distance field from an outline
*/
FS_GRAYMAP* make_distance_field(_DS_ FS_OUTLINE *outl, FS_ULONG id)
{
    ADFPath *a;
    void *c;
    ADFImage* di;
    ADFRenderImageAttrs adfRenderImageAttrs;
    ADFTypesetAttrs adfTypesetAttrs;
    FS_GRAYMAP *g = 0;
    FS_FIXED scaled_dx, scaled_dy;

    adfRenderImageAttrs.displayMode = ADF_REND_MODE_CRT;

    a = make_ADF_path(_PS_ outl, id, &g, &scaled_dx, &scaled_dy);
    if (!a)  /* we could have a space character */
        return g;  /* which will be in g */

    c = make_ADF_char(_PS_ a);
    if (!c)          /* STATE contains error code   */
        return g;   /* a is freed in make_ADF_char */

    di = make_ADF_density_image(_PS_ c, &adfRenderImageAttrs, &adfTypesetAttrs);
    if (!di)         /* STATE contains error code            */
        return g;   /* c is freed in make_ADF_density_image */

    /* convert to 8-bit graymap initially */
    g  = convert_to_distance_field(_PS_ di, scaled_dx, scaled_dy,
                                   &adfRenderImageAttrs, &adfTypesetAttrs);

    return g;
}

/**
* Private function to get a distance field
*/
FS_GRAYMAP *get_distance_field(_DS_ FS_ULONG id, FS_USHORT index, FS_SHORT phasex, FS_SHORT phasey)
{
    FS_GRAYMAP *gmap;
    FS_OUTLINE *outl;
    LFNT *lfnt = STATE.cur_lfnt;
    SFNT *sfnt = STATE.cur_sfnt;
    FS_FIXED insideCutoff, outsideCutoff, cutoff;
    FS_ULONG flags;

    STATE.error = SUCCESS;

    if (lfnt==0 || sfnt==0)
        return 0;

    /* save current cutoff values and state flag */
    insideCutoff  = STATE.insideCutoff;
    outsideCutoff = STATE.outsideCutoff;
    flags = STATE.flags;

    /* compute appropriate CSM values to get a distance field */
    cutoff = FIXED_ONE + 10486*STATE.lpm; /* 1.0 + 0.16*STATE.lpm */
    FS_set_csm(_PS_ cutoff, -cutoff, FIXED_ONE);
    FS_set_flags(_PS_ FLAGS_DEFAULT_CSM_ON);
    STATE.flags |= FLAGS_GRAYSCALE;  /* use grayscale autohinting if applicable */

#ifdef FS_EXTERNAL_OUTLINE
    if (STATE.user_outline)
    {
        outl = copy_outline(_PS_ STATE.user_outline, 0 /* source is in server memory */);
        if (!outl)
        {
            STATE.insideCutoff = insideCutoff;
            STATE.outsideCutoff = outsideCutoff;
            STATE.flags = flags;
            return 0;
        }

        /* possibly shift outline by quarter-pixel amount */
        shift_outline(outl, phasex, phasey);

        /* then make the graymap ... */
        gmap = make_distance_field(_PS_ outl, id);

        STATE.insideCutoff = insideCutoff;
        STATE.outsideCutoff = outsideCutoff;
        STATE.flags = flags;
        FSS_free_char(_PS_ outl);

        return ((gmap == 0 || STATE.error) ? 0 : gmap);
    }
#endif /* FS_EXTERNAL_OUTLINE */

#ifdef FS_CACHE_EDGE_GLYPHS
    /* first check whether distance field has been cached */
    gmap = find_graymap_in_cache(_PS_ index, CACHE_ADF_GRAYMAP8, phasex, phasey);
    if (gmap)
    {
        STATE.insideCutoff = insideCutoff;
        STATE.outsideCutoff = outsideCutoff;
        STATE.flags = flags;
        return gmap;
    }
#endif /* FS_CACHE_EDGE_GLYPHS */

    outl = find_or_make_outline(_PS_ lfnt, sfnt, id, index);
    if (!outl)
        return 0;

    FS_set_flags(_PS_ FLAGS_CMAP_OFF);  /* already have index */

    /* possibly shift outline by quarter-pixel amount */
    shift_outline(outl, phasex, phasey);

    gmap = make_distance_field(_PS_ outl, id);
    if (STATE.error)
    {
        STATE.insideCutoff = insideCutoff;
        STATE.outsideCutoff = outsideCutoff;
        STATE.flags = flags;
        FSS_free_char(_PS_ outl);
        return 0;
    }

#ifdef FS_CACHE_EDGE_GLYPHS
    save_graymap_to_cache(_PS_ index, gmap, CACHE_ADF_GRAYMAP8, phasex, phasey);
#endif
#ifdef FS_CACHE_OUTLINES
    /* restore the cached outline to its original unshifted state */
    shift_outline(outl, -phasex, -phasey);
#endif

    STATE.insideCutoff = insideCutoff;
    STATE.outsideCutoff = outsideCutoff;
    STATE.flags = flags;
    FSS_free_char(_PS_ outl);

    return gmap;
}

/**
* Private function to get the adjusted ADF graymap advance given an outline
* Adjustment only applies to rotated glyphs
* Does most but not all of the work of make_ADF_graymap
*/
FS_VOID get_ADF_adjustments(_DS_ FS_OUTLINE *outl, FS_ULONG id,
                            FS_FIXED *adj_x, FS_FIXED *adj_y)
{
    ADFPath *a;
    void *c;
    FS_GRAYMAP *g = 0;
    ADFRenderAttrs adfRenderAttrs;
    ADFRenderImageAttrs adfRenderImageAttrs;
    ADFTypesetAttrs adfTypesetAttrs;
    ADFRenderGlyphData adfRenderGlyphData;
    FS_FIXED scaled_dx, scaled_dy;
    FS_OUTLINE *tmp=0;

#ifdef FS_STIK
    int is_stik = 0;
#endif

    *adj_x = 0;
    *adj_y = 0;

    if (STATE.adfGridFitType == ADF_GRID_FIT_NONE)
        return;

    if (outl == 0)
        return;

#ifdef FS_STIK
    if (outl->num != 0)
        is_stik = !(((FS_BYTE *)(outl->type))[0] & OUTLINE_CHAR);

    if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK)
        && is_stik)
    {
#ifdef FS_PSEUDO_BOLD
        if (STATE.bold_pct || STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct)
        {
            if ( (STATE.flags & (FLAGS_EMBOSSED | FLAGS_ENGRAVED | FLAGS_OUTLINED_1PIXEL |
                                FLAGS_OUTLINED_2PIXEL | FLAGS_OUTLINED_UNFILLED)) ||
                ( !(STATE.flags & FLAGS_ADD_WEIGHT) &&
                ( !( STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) || STATE.lpm > 26 ) )
               )
            {
                FS_LONG n_types, n_points;

                outl = expand_stik(_PS_ outl,&n_types,&n_points);
                if (!outl)
                    return;

                tmp = embolden_outline(_PS_ outl,&n_types,&n_points);
                if (!tmp)
                {
                    return;
                }

                if (tmp != outl)
                    FSS_free_char(_PS_ outl);
                outl = tmp;
            }
        }
#endif /* FS_PSEUDO_BOLD */
    }
#endif /* FS_STIK */

    a = make_ADF_path(_PS_ outl, id, &g, &scaled_dx, &scaled_dy);
    if (!a)  /* we could have a space character */
    {
        if (g)
            FSS_free(_PS_ g);
        return;
    }

    c = make_ADF_char(_PS_ a);
    if (!c)
        return;

    adfCopyRenderAttrs(&STATE.cur_typeset.tfntarray[STATE.cur_font].adfRenderAttrs, 
                       &adfRenderAttrs);

    if (STATE.flags & FLAGS_DEFAULT_CSM_ON)
    {    /* override font values in STATE copy */
        adfRenderAttrs.insideCutoff = STATE.insideCutoff + STATE.insideCutoffAdj;
        adfRenderAttrs.outsideCutoff = STATE.outsideCutoff + STATE.outsideCutoffAdj;
        adfRenderAttrs.gamma = STATE.gamma;
    }

    if (adfRenderAttrs.insideCutoff < adfRenderAttrs.outsideCutoff)
    {   /* clamp cutoffs to prevent inverted cutoffs */
        if (adfRenderAttrs.outsideCutoff < 0)
        {   /* thick binary output */
            adfRenderAttrs.insideCutoff = adfRenderAttrs.outsideCutoff;
        }
        else
        {   /* regular binary output */
            adfRenderAttrs.insideCutoff = 0;
            adfRenderAttrs.outsideCutoff = 0;
        }
    }

    /* Use adfRenderAttrs to fill in the other 3 structures */
    ADFRenderSetup(c,
                   &adfRenderAttrs,
                   &adfRenderGlyphData,
                   &adfRenderImageAttrs,
                   &adfTypesetAttrs);

    ADFDestroyADF(STATE.libInst, c);

    if (scaled_dx && scaled_dy)
    {
        *adj_x = FixMul(scaled_dx,adfTypesetAttrs.adjFUToPixelScaleX) + 
                 (FS_FIXED)(adfTypesetAttrs.gridAlgnAdjX) - scaled_dx;
        *adj_y = FixMul(scaled_dy,adfTypesetAttrs.adjFUToPixelScaleY) + 
                 (FS_FIXED)(adfTypesetAttrs.gridAlgnAdjY) - scaled_dy;
    }
    if (tmp)
        FSS_free_char(_PS_ tmp);
}


/**
* Private function to make an ADF graymap from scratch given an outline
*/
FS_GRAYMAP* make_ADF_graymap(_DS_ FS_OUTLINE *outl, FS_ULONG id, FS_USHORT type)

{
    ADFPath *a;
    void *c;
    ADFImage* di;
    ADFRenderImageAttrs adfRenderImageAttrs;
    ADFTypesetAttrs adfTypesetAttrs;
    FS_GRAYMAP *g = 0;
    FS_FIXED scaled_dx, scaled_dy;
    FS_BOOLEAN needs_trim_convert = 1;

#ifdef FS_STIK
    int is_stik = 0;
    FS_BOOLEAN free_outl = 0;

    if (outl->num != 0)
        is_stik = !(((FS_BYTE *)(outl->type))[0] & OUTLINE_CHAR);

    if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK)
        && is_stik)
    {
#ifdef FS_PSEUDO_BOLD
        if (STATE.bold_pct || STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct)
        {
            FS_LONG n_types, n_points;
            FS_OUTLINE *tmp;

            if ( (STATE.flags & FLAGS_OUTLINEBOLD)   ||
                 ( !(STATE.flags & FLAGS_ADD_WEIGHT) &&
                 ( !( STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) || STATE.lpm > 26)) )
            {
                free_outl = 1;
                outl = expand_stik(_PS_ outl,&n_types,&n_points);
                if (!outl)
                    return g;

                tmp = embolden_outline(_PS_ outl,&n_types,&n_points);
                if (!tmp)
                {
                    if (free_outl)
                        FSS_free_char(_PS_ outl);
                    return g;
                }

                if (tmp != outl)
                    FSS_free_char(_PS_ outl);
                outl = tmp;
            }
        }
        else
#endif /* FS_PSEUDO_BOLD */
        {
            if ( !( STATE.cur_lfnt->fontflags & FONTFLAG_EDGE) &&
                 !( STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON))
            {
                FS_OUTLINE *cp_outl;
                cp_outl = copy_outline(_PS_ outl, 0);
                if (cp_outl)
                {
                    free_outl = 1;
                    outl = cp_outl;
                }
                autohint_stik(_PS_ outl, (FS_FIXED)(STATE.stroke_pct * STATE.lpm));
            }
        }
    }
#endif /* FS_STIK */
    if (type & ~FS_MAP_ANY_EDGE_SUBPIXEL)
    {
        adfRenderImageAttrs.displayMode = ADF_REND_MODE_CRT;
        type &= ~FS_MAP_ANY_EDGE_SUBPIXEL;
    }
    else if (type & FS_MAP_ANY_EDGE_SUBPIXEL)
    {
        if (type & (FS_MAP_EDGE_RGBv4|FS_MAP_EDGE_RGBv8))
            adfRenderImageAttrs.displayMode = ADF_REND_MODE_RGBv;
        else
            adfRenderImageAttrs.displayMode = ADF_REND_MODE_RGBh;
        type &= FS_MAP_ANY_EDGE_SUBPIXEL;
    }

    a = make_ADF_path(_PS_ outl, id, &g, &scaled_dx, &scaled_dy);
    if (!a)  /* we could have a space character */
        return g;  /* which will be in g */

    c = make_ADF_char(_PS_ a);
    if (!c)          /* STATE contains error code   */
        return g;   /* a is freed in make_ADF_char */

    di = make_ADF_density_image(_PS_ c, &adfRenderImageAttrs, &adfTypesetAttrs);
    if (!di)         /* STATE contains error code            */
        return g;   /* c is freed in make_ADF_density_image */

    if (type & FS_MAP_ANY_EDGE_SUBPIXEL)
    {
        g  = convert_to_graymap(_PS_ di, scaled_dx, scaled_dy, 
                                    &adfRenderImageAttrs, &adfTypesetAttrs, type);
#ifdef FS_STIK        
        if (free_outl) 
            FSS_free_char(_PS_ outl);
#endif  
        if ( g == 0 )
            return 0;

        g = trim_subpixel(_PS_ g, type);
    }
    else
    {
        /* convert to 8-bit graymap initially */
        g  = convert_to_graymap(_PS_ di, scaled_dx, scaled_dy,
                                &adfRenderImageAttrs, &adfTypesetAttrs, FS_MAP_EDGE_GRAYMAP8);
        if (g == 0)
        {
#ifdef FS_STIK
            if (free_outl) 
                FSS_free_char(_PS_ outl);
#endif
            return 0;
        }

        /* reset i_dx,i_dy according to outl */
        if (outl->i_dx == 0 )
            g->i_dx = 0;
        if (outl->i_dy == 0)
            g->i_dy = 0;

#ifdef FS_STIK
        if (free_outl)
            FSS_free_char(_PS_ outl);
#endif

        /* Apply any special effects */
        /* 2-bit output looks better if we convert first */
        if (type == FS_MAP_EDGE_GRAYMAP2)
        {
            g = trim_convert(_PS_ g, type);
            needs_trim_convert = 0;
        }

#ifdef FS_PSEUDO_BOLD
        if ((STATE.bold_pct || STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct))
        {
            if ((STATE.flags & FLAGS_ADD_WEIGHT) ||
                ((STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) &&
                 (STATE.lpm <= 26) && !(STATE.flags & FLAGS_OUTLINEBOLD)
                 ))
            {
                if (g->bitsPerPixel == 8)
                    pixelbold_graymap8(_PS_ g);
                else if (g->bitsPerPixel == 2)
                    pixelbold_graymap2(_PS_ g);
            }
        }
#endif

        if (STATE.flags & FLAGS_EMBOSSED)
            g = emboss_graymap(g);
        else if (STATE.flags & FLAGS_ENGRAVED)
            g = engrave_graymap(g);

        /* n-pixel outline effect */
        if (STATE.flags & FLAGS_OUTLINED)
            g = outline_graymap(_PS_ g, 1, STATE.outline_opacity);
        else if (STATE.flags & FLAGS_OUTLINED_2PIXEL)
            g = outline_graymap(_PS_ g, 2, STATE.outline_opacity);
        else if ((STATE.flags & FLAGS_OUTLINED_FILLED)||
                 (STATE.flags & FLAGS_OUTLINED_UNFILLED))
            g = outline_graymap(_PS_ g, STATE.outline_width, STATE.outline_opacity);
        else if (STATE.flags & FLAGS_OUTLINED_SOFT)
            g = soft_outline_graymap(_PS_ g, STATE.outline_width, STATE.outline_opacity);

        /* soften effect */
        if ((STATE.flags & FLAGS_SOFTENED))
        {
            /* default soften filter parameters differ by size, but          */
            /* your system or personal preferences may differ ... experiment */
            if (g)
            {
                if (STATE.lpm <17)
                    g = Soften(_PS_ 1, 1, g);
                else if (STATE.lpm <26)
                    g = Soften(_PS_ 2, 2, g);
                else
                    g = Soften(_PS_ 3, 3, g);
            }
        }
        if (STATE.error)
        {
            FSS_free_char(_PS_ g);  /* just in case */
            return 0;
        }

        /* default to 4-bit Edge graymap if any Edge graymap specified */
        if (type == FS_MAP_ANY_EDGE_GRAYMAP)
            type = FS_MAP_EDGE_GRAYMAP4;

        if (g && needs_trim_convert)
            g = trim_convert(_PS_ g, type);
    }

#ifdef FS_STATS
    made_gmap++;
#endif

    return g;
}

/**
* Private function to get an ADF graymap, either from cache or from scratch.
* This function is called from FS_get_glyph when a ADF graymap is requested.
* Since we do not use SAZ data (deprecated), access is through glyph index.
*/
FS_GLYPHMAP *get_ADF_graymap(_DS_ FS_ULONG id, FS_USHORT index, FS_USHORT type, 
                             FS_SHORT phasex, FS_SHORT phasey)
{
    LFNT *lfnt;
    TTF *ttf;
    TFNT *tfnt = &STATE.cur_typeset.tfntarray[STATE.cur_font];
    FS_ULONG flags;
    FS_GRAYMAP *gmap;
#ifdef FS_CACHE_EDGE_GLYPHS
    FS_SHORT cache_type = 0;
#endif /* FS_CACHE_EDGE_GLYPHS */

    STATE.error = SUCCESS;

    if (type & FS_MAP_ANY_EDGE_SUBPIXEL)
    {
        if  ((STATE.flags & (~FLAGS_NO_EFFECT)) || 
             (STATE.flags & (~FLAGS_NO_GRAYMAP_EFFECT)) || 
             (STATE.flags & (FLAGS_ADD_WEIGHT_ON)))
        {
              STATE.error = ERR_EFFECT_UNSUPPORTED;
              return 0;
        }
    } 
#ifdef FS_EXTERNAL_OUTLINE
    if (STATE.user_outline)
    {
        FS_OUTLINE *outl;

        outl = copy_outline(_PS_ STATE.user_outline,
                            0 /* source is in server memory */);
        if (!outl) return 0;

        /* possibly shift outline by quarter-pixel amount */
        shift_outline(outl, phasex, phasey);

        /* then make the graymap ... */
        gmap = make_ADF_graymap(_PS_ outl, 0, type);
        if (gmap == 0 || STATE.error)
        {
            FSS_free_char(_PS_ outl);
            return 0;
        }
        FSS_free_char(_PS_ outl);

        return (FS_GLYPHMAP *)gmap;
    }
#endif /* FS_EXTERNAL_OUTLINE */

    lfnt = STATE.cur_lfnt;
    if (lfnt==0)
        return 0;

    /* get CSM values from the adfh table, if present */
    ttf = (TTF *)(lfnt->fnt);
    if (!ttf)
    {
        return 0;
    }

    /* get the CSM data for this glyph index from the adfh table   */
    /* If the font does not contain CSM value for this glyph index */
    /* and scale, use the defaults in STATE. Otherwise search the  */
    /* ADFH table for the appropriate CSM values based on scale.   */
    adfGetCSMValues(_PS_ ttf,index,&tfnt->adfRenderAttrs);

    /* apply any user adjustments */
    tfnt->adfRenderAttrs.insideCutoff  += STATE.insideCutoffAdj;
    tfnt->adfRenderAttrs.outsideCutoff += STATE.outsideCutoffAdj;

    STATE.flags |= FLAGS_GRAYSCALE;  /* use grayscale autohinting if applicable */

#ifdef FS_CACHE_EDGE_GLYPHS
    if (type & FS_MAP_EDGE_GRAYMAP4)
        cache_type = CACHE_ADF_GRAYMAP;
    else if (type & FS_MAP_EDGE_GRAYMAP8)
        cache_type = CACHE_ADF_GRAYMAP8;
    else if (type & FS_MAP_EDGE_GRAYMAP2)
        cache_type = CACHE_ADF_GRAYMAP2;
    else if (type & FS_MAP_EDGE_RGBv4)
        cache_type = CACHE_ADF_RGBv4;
    else if (type & FS_MAP_EDGE_RGBv8)
        cache_type = CACHE_ADF_RGBv8;
    else if (type & FS_MAP_EDGE_RGBh4)
        cache_type = CACHE_ADF_RGBh4;
    else if (type & FS_MAP_EDGE_RGBh8)
        cache_type = CACHE_ADF_RGBh8;
    gmap = find_graymap_in_cache(_PS_ index,cache_type,phasex,phasey);
    if (gmap)
        return (FS_GLYPHMAP *)gmap;
#else
    gmap = 0;
#endif /* FS_CACHE_EDGE_GLYPHS */


#ifdef FS_EMBEDDED_BITMAP
    /* check for embedded graymap */
    if ((lfnt->fnt_type == TTF_TYPE) || (lfnt->fnt_type == TTC_TYPE))
    {
        SFNT *sfnt = STATE.cur_sfnt;

        gmap = get_embedded_graymap(_PS_ sfnt,(FS_USHORT)index, type);
        if (STATE.error)
            return 0;

        if (gmap)
        {
#ifdef FS_PSEUDO_BOLD
            {
                FS_SHORT bw = sfnt->senv->bold_width;
                if (bw)
                {
                    gmap = pixelbold_embedded_graymap(_PS_ gmap);
                }
            }
#endif /* FS_PSEUDO_BOLD */
            /* emboss/engrave effect */
            if (STATE.flags & FLAGS_EMBOSSED)
                gmap = emboss_graymap(gmap);
            else if (STATE.flags & FLAGS_ENGRAVED)
                gmap = engrave_graymap(gmap);

            /* n-pixel outline effect */
            if (STATE.flags & FLAGS_OUTLINED)
                gmap = outline_graymap(_PS_ gmap, 1, STATE.outline_opacity);
            else if (STATE.flags & FLAGS_OUTLINED_2PIXEL)
                gmap = outline_graymap(_PS_ gmap, 2, STATE.outline_opacity);
            else if ((STATE.flags & FLAGS_OUTLINED_FILLED)||
                        (STATE.flags & FLAGS_OUTLINED_UNFILLED))
                gmap = outline_graymap(_PS_ gmap, STATE.outline_width,
                                            STATE.outline_opacity);    /* soft outline effect, regular outlines are done on density image */
            else if ((STATE.flags & FLAGS_OUTLINED_SOFT))
                gmap = soft_outline_graymap(_PS_ gmap, STATE.outline_width,
                                            STATE.outline_opacity);

            /* soften effect */
            if ((STATE.flags & FLAGS_SOFTENED))
            {
                /* default soften filter parameters differ by size, but          */
                /* your system or personal preferences may differ ... experiment */
                if (gmap)
                {
                    if (STATE.lpm <17)
                        gmap = Soften(_PS_ 1, 1, gmap);
                    else if (STATE.lpm <26)
                        gmap = Soften(_PS_ 2, 2, gmap);
                    else
                        gmap = Soften(_PS_ 3, 3, gmap);
                }
            }
#ifdef FS_CACHE_EDGE_GLYPHS
            /* if font is disk based, cache the embedded graymap */
            /* if gmap had special effects applied, cache it as well */
            if (lfnt->path || (STATE.flags & SPECIALEFFECTS))
                save_graymap_to_cache(_PS_ index,gmap,cache_type,phasex,phasey);
#endif /* FS_CACHE_EDGE_GLYPHS */
            return (FS_GLYPHMAP *)gmap;
        }
    }
#endif  /* FS_EMBEDDED_BITMAP */

    /* make it from scratch */
    if (!gmap)
    {
        FS_OUTLINE *outl;
        SFNT *sfnt = STATE.cur_sfnt;
        if (sfnt==0)
            return 0;

        outl = find_or_make_outline(_PS_ lfnt, sfnt, id, index);
        if (!outl)
            return 0;

        /* possibly shift outline by quarter-pixel amount */
        shift_outline(outl, phasex, phasey);

        flags = STATE.flags;
        FS_set_flags(_PS_ FLAGS_CMAP_OFF);  /* already have index */
        gmap = make_ADF_graymap(_PS_ outl, id, type);

        STATE.flags = flags;
        if (STATE.error)
        {
            FSS_free_char(_PS_ outl);
            return 0;
        }

#ifdef FS_CACHE_OUTLINES
        /* restore the cached outline to its original unshifted state */
        shift_outline(outl, -phasex, -phasey);
#endif

        FSS_free_char(_PS_ outl);

        if (gmap == 0)
            return 0;

        /* apply any vertical shift adjustment for component font */
        if ( sfnt && sfnt->vertical_shift)
        {
            SENV *senv = (SENV *)(sfnt->senv);

            if (senv)  /* should be true always at this point */
            {
                if (senv->vanilla)
                {
                    gmap->hi_y += sfnt->vertical_shift;
                }
                else
                {
                    FIXED_VECTOR p;
                    p.x = gmap->dx;
                    p.y = gmap->dy;
                    fixed_norm(&p);  /* unit tangent vector */
                    /* adjust graymap placement by scaled normal vector */
                    gmap->lo_x += (FS_SHORT)FixMul(-p.y,sfnt->vertical_shift);
                    gmap->hi_y += (FS_SHORT)FixMul( p.x,sfnt->vertical_shift);
                }
            }
        }

        if (STATE.error)
            return 0;
    }

    if (STATE.error)
    {
        FSS_free_char(_PS_ gmap);
        return 0;
    }
#ifdef FS_CACHE_EDGE_GLYPHS
    save_graymap_to_cache(_PS_ index,gmap,cache_type,phasex,phasey);
#endif

    return (FS_GLYPHMAP *)gmap;
}

#endif /* FS_EDGE_RENDER */
